/**
* \file: socket_server.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: automounter
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2010, 2011 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#include <string.h>
#include <stdlib.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <sys/poll.h>
#include <stdbool.h>
#include <signal.h>
#include <unistd.h>
#include <errno.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "app_iface/socket_server.h"
#include "utils/logger.h"
#include "utils/automounter_types_internal.h"
#include "utils/global_constants.h"
#include "control/daemon_constants.h"
#include "control/watch.h"
#include "control/automounter.h"
#include "control/configuration.h"
#include "ipc/info_messages.h"
#include "ipc/message_sendr.h"
#include "ipc/message_recvr.h"

#undef SUN_LEN
#define SUN_LEN(su) ((sizeof(*(su)) - sizeof((su)->sun_path)) + strlen((su)->sun_path))


typedef struct connection_ctx_t connection_ctx_t;

typedef struct connection_ctx_t
{
	char *app_identifier;
	event_mask_t event_mask;
	watch_t watch;
	message_queue_t sender_queue;
	message_buffer_t *receiver_buffer;
	connection_ctx_t *next_connection;
} connection_ctx_t;

static void socket_server_handle_incoming_connection(watch_t *ev_watch, uint32_t events);

static void socket_server_handle_connection_event(watch_t *watch, uint32_t events);
static void socket_server_handle_data_on_connection(watch_t *watch);
static void socket_server_handle_connection_free_for_sending(watch_t *watch);

static void socket_server_disable_connection_pollout_watch(connection_ctx_t *con);
static void socket_server_enable_connection_pollout_watch(connection_ctx_t *con);

static void socket_server_add_connection(int con_socket);

static void socket_server_remove_connection(connection_ctx_t *con);

static void socket_server_on_connection_lost(connection_ctx_t *con);

static void socket_server_close_all_connections(void);


static connection_ctx_t *socket_server_create_connection_ctx(int con_socket);

static void socket_server_connection_close_and_free_ctx(connection_ctx_t *con);


// singleton module attributes
static watch_t listening_socket_watch={-1,0,socket_server_handle_incoming_connection,NULL};

static connection_ctx_t connection_list_head={NULL,0,{0},{0},NULL,NULL};

static socket_server_handle_msg_callback_t socket_server_msg_callback_func=NULL;


//------------------------------------------ API members -------------------------------------------------
error_code_t socket_server_init(socket_server_handle_msg_callback_t msg_callback_func)
{
	error_code_t result;
	struct sockaddr_un serveraddr;

	if (listening_socket_watch.pollfd!=-1)
		return RESULT_INVALID;

	socket_server_msg_callback_func=msg_callback_func;
	//initialize message sendr module
	result=message_sendr_init();

	if (result!=RESULT_OK)
		return result;

	//creating a socket
	listening_socket_watch.pollfd=socket(AF_UNIX,SOCK_STREAM|SOCK_NONBLOCK,0);
	if (listening_socket_watch.pollfd<0)
	{
		logger_log_error("SOCKET_SERVER - Error creating unix socket.\n");
		result=RESULT_SOCKET_ERR;
	}

	//bind it to a address
	if (result==RESULT_OK)
	{
		memset(&serveraddr, 0, sizeof(serveraddr));
		serveraddr.sun_family = AF_UNIX;
		strcpy(serveraddr.sun_path, AUTOMOUNTER_CTRL_SOCKET);

		if(bind(listening_socket_watch.pollfd, (struct sockaddr *)&serveraddr, SUN_LEN(&serveraddr))<0)
		{
			logger_log_error("SOCKET_SERVER - Error binding the control socket to: %s\n",
			        AUTOMOUNTER_CTRL_SOCKET);
			result=RESULT_SOCKET_ERR;
		}
	}

	//at this point, the socket in the file system is already created but not
	//right to accept connections. So we can savely set the access rights at this point
	if (result==RESULT_OK)
	{
		(void)chmod(AUTOMOUNTER_CTRL_SOCKET, configuration_get_socket_access_mode());
		if (chown(AUTOMOUNTER_CTRL_SOCKET, getuid(), configuration_get_socket_group_id()))
		{

		}
	}

	//and start listening
	if (result==RESULT_OK)
	{
		if(listen(listening_socket_watch.pollfd, AUTOMOUNTER_CTL_NUM_PARALLEL_CONNECTIONS)< 0)
		{
			logger_log_error("SOCKET_SERVER - Error start listening to the control socket.\n",AUTOMOUNTER_CTRL_SOCKET);
			result=RESULT_SOCKET_ERR;
		}
	}

	if (result==RESULT_OK)
		result=watch_add_event_source(&listening_socket_watch,EPOLLIN);

	if (result==RESULT_OK)
	{
		int ready_flag_file_fd;
		ready_flag_file_fd=open(AUTOMOUNTER_SOCKET_READY_FILE,O_CREAT, 0);
		if (ready_flag_file_fd<0)
		{
			logger_log_error("SOCKET_SERVER - Unable to create the socket ready flag file: %s",
					AUTOMOUNTER_SOCKET_READY_FILE);
			result=RESULT_INVALID;
		}
		else
			close(ready_flag_file_fd);

		(void)chmod(AUTOMOUNTER_SOCKET_READY_FILE, configuration_get_socket_access_mode());
		if (chown(AUTOMOUNTER_SOCKET_READY_FILE, getuid(), configuration_get_socket_group_id()))
		{

		}
	}

	if (result==RESULT_OK)
		logger_log_debug("SOCKET_SERVER - Now listening to control socket: %s",AUTOMOUNTER_CTRL_SOCKET);

	return result;
}

message_buffer_t *socket_server_get_empty_msg_buffer(void)
{
	return message_sendr_get_empty_buffer();
}

error_code_t socket_server_send_msg(message_buffer_t *msg_buffer, connection_ctx_t *connection)
{
	error_code_t result;
	result=message_sendr_send_buffer(msg_buffer, connection->watch.pollfd);
	if (result==RESULT_SOCKET_ERR)
	{
		logger_log_error("SOCKET_SERVER - Connection lost to client '%s' while proceeding a request."
				" Closing the connection.", connection->app_identifier);
		socket_server_remove_connection(connection);
	}
	else if (result==RESULT_MSG_PARTLY_PROCESSED)
	{
		message_sendr_add_buffer_to_send_queue(msg_buffer, &(connection->sender_queue));
		socket_server_enable_connection_pollout_watch(connection);
		logger_log_debug("SOCKET_SERVER - Socket buffer of connection to client %s full. "
				"Put message to send into the queue.", connection->app_identifier);
		return RESULT_OK;
	}

	// in case of problem with the socket or if we could successfully send the buffer, we are putting it back into the
	// pool of empty buffers
	message_sendr_putback_buffer(msg_buffer);

	return result;
}

void socket_server_broadcast_msg(message_buffer_t *msg_buffer, event_mask_t event)
{
    connection_ctx_t *con;
	error_code_t result;
    con=connection_list_head.next_connection;
    while(con!=NULL)
    {
    	if (socket_server_is_ctx_registered_for_event(con,event))
    	{
    		result=message_sendr_send_buffer(msg_buffer, con->watch.pollfd);
    		if (result==RESULT_SOCKET_ERR)
    		{
    			logger_log_error("SOCKET_SERVER - Connection lost to client '%s' while proceeding a request."
    					" Closing the connection.", con->app_identifier);
    			socket_server_remove_connection(con);
    		}
    		else if (result==RESULT_MSG_PARTLY_PROCESSED)
    		{
    			// we need to queue the buffer into one of the sender queues
    			// 1. get a new empty buffer
    			// 2. take a copy of the one we need to queue
    			// 3. queue the buffer
    			// 4. go on with the copy of the buffer
    			message_buffer_t *new_buffer;
    			//1.
    			new_buffer=message_sendr_get_empty_buffer();
    			if (new_buffer==NULL)
    				automounter_die_on_resource_issues();
    			//2.
    			message_buffer_clone_msg_in_buffer(new_buffer,msg_buffer);
    			//3.
    			message_sendr_add_buffer_to_send_queue(msg_buffer, &(con->sender_queue));
    			socket_server_enable_connection_pollout_watch(con);
    			//4.
    			msg_buffer=new_buffer;
    			result=RESULT_OK;
    			logger_log_debug("SOCKET_SERVER - Socket buffer of connection to client %s full. "
    					"Put message to send into the queue.", con->app_identifier);
    		}

    		//reset the buffer keeping the message inside to use it for sending again
    		message_buffer_reuse_message_for_sending(msg_buffer);
    	}
        con=con->next_connection;
    }

    //we can put back the buffer in any case since we took a new buffer any time we queued one into a sender queue
	message_sendr_putback_buffer(msg_buffer);
}

bool socket_server_is_ctx_registered_for_event(connection_ctx_t *ctx, event_mask_t event)
{
	return (ctx->event_mask & event) == event;
}

void socket_server_deinit(void)
{
	if (listening_socket_watch.pollfd==-1)
		return;

	logger_log_debug("SOCKET_SERVER - Closing control socket.");
	unlink(AUTOMOUNTER_SOCKET_READY_FILE);
	socket_server_close_all_connections();
	watch_remove_event_source(&listening_socket_watch);
	close(listening_socket_watch.pollfd);
	listening_socket_watch.pollfd=-1;
	unlink(AUTOMOUNTER_CTRL_SOCKET);
	message_sendr_deinit();
}

void socket_server_set_connection_appinfo(connection_ctx_t *ctx, const char *app_identifier, event_mask_t event_mask)
{
	ctx->event_mask=event_mask;
	if (app_identifier != NULL)
		ctx->app_identifier=strdup(app_identifier);
	//else
		//we initialized app_identifier with "unknown application" before. We keep this value if no identifier has been
		//send to us
	if (ctx->app_identifier==NULL)
	{
		logger_log_error("MESSAGE_DISPATCHER - Resource problems after getting information about application: %s",app_identifier);
		automounter_die_on_resource_issues();
	}
}

const char *socket_server_get_connection_appidentifier(connection_ctx_t *ctx)
{
	return ctx->app_identifier;
}

event_mask_t socket_server_get_connection_event_mask(connection_ctx_t *ctx)
{
	return ctx->event_mask;
}

message_buffer_t *socket_server_get_receiver_buffer(connection_ctx_t *ctx)
{
	if (ctx != NULL)
		return ctx->receiver_buffer;
	else
		return NULL;
}
//--------------------------------------------------------------------------------------------------------

//------------------------------------------ private members ---------------------------------------------
static void socket_server_handle_incoming_connection(watch_t *ev_watch, uint32_t events)
{
	int con_socket;

	//we know our watch
	(void)ev_watch;
	//we registered only for EPOLLIN
	(void)events;

	logger_log_debug("SOCKET_SERVER - Incoming connection.");

	con_socket=accept(listening_socket_watch.pollfd,NULL,NULL);
	if (con_socket<0)
	{
		logger_log_error("SOCKET_SERVER - Unable to accept incoming connection. Ignoring it.");
		return;
	}

	if (fcntl(con_socket,F_SETFL,O_NONBLOCK)!=0)
	{
		logger_log_error("SOCKET_SERVER - Unable to set the new socket to non blocking mode. Closing the connection.");
		close(con_socket);
		return;
	}

	socket_server_add_connection(con_socket);
    logger_log_debug("SOCKET_SERVER - Connection established.");
}

static void socket_server_handle_connection_event(watch_t *watch, uint32_t events)
{
	if ((events & EPOLLIN)!=0)
		socket_server_handle_data_on_connection(watch);
	if ((events & EPOLLOUT)!=0)
		socket_server_handle_connection_free_for_sending(watch);
}

static void socket_server_handle_connection_free_for_sending(watch_t *watch)
{
	error_code_t result;
    connection_ctx_t *con;
    con=(connection_ctx_t *)watch->data;
	logger_log_debug("SOCKET_SERVER - Got an event for connection to client '%s' that the socket is now able to receive"
			" data again. Sending queued messages now.", con->app_identifier);

	result=message_sendr_send_queued_buffers(&(con->sender_queue),con->watch.pollfd);

	if (result != RESULT_MSG_PARTLY_PROCESSED)
		// we either managed freeing our queue or we run into a sending error. In both cases we can
		// remove the pollout event from our watch
		socket_server_disable_connection_pollout_watch(con);
	else
		logger_log_debug("SOCKET_SERVER - Socket buffer of connection to client %s full. "
				"Buffers remained in the queue. Sending them later.", con->app_identifier);


	if (result == RESULT_SOCKET_ERR)
	{
		logger_log_error("SOCKET_SERVER - Connection lost to client '%s' while proceeding a request."
				" Closing the connection.", con->app_identifier);
		socket_server_remove_connection(con);
	}
}

static void socket_server_disable_connection_pollout_watch(connection_ctx_t *con)
{
	watch_modify_watch_events(&(con->watch), EPOLLIN);
}

static void socket_server_enable_connection_pollout_watch(connection_ctx_t *con)
{
	watch_modify_watch_events(&(con->watch), EPOLLIN|EPOLLOUT);
}

static void socket_server_handle_data_on_connection(watch_t *watch)
{
	error_code_t result;
    connection_ctx_t *con;
    con=(connection_ctx_t *)watch->data;

    result=message_recvr_receive_msg(con->receiver_buffer,con->watch.pollfd);
	if (result==RESULT_SOCKET_CLOSED || result==RESULT_SOCKET_ERR)
		socket_server_on_connection_lost(con);
	else if (result==RESULT_MSG_PARTLY_PROCESSED)
		logger_log_debug("SOCKET_SERVER - Received first part of a message from application '%s'. Waiting for the rest!!",
				socket_server_get_connection_appidentifier(con));
	else if (result==RESULT_SOCKET_MSG_CORRUPT)
		logger_log_error("SOCKET_SERVER - Received a corrupt message from application '%s'. Ignoring it!!",
				socket_server_get_connection_appidentifier(con));
	else if (result==RESULT_INVALID)
		logger_log_error("SOCKET_SERVER - We tried to read data from a connection without having initialized the "
				"generic message module before. Implementation error!!");
	else if (result==RESULT_NORESOURCE)
		automounter_die_on_resource_issues();
	else if (result==RESULT_OK)
	{
		logger_log_debug("SOCKET_SERVER - Received a message from application '%s'.",
				socket_server_get_connection_appidentifier(con));
		socket_server_msg_callback_func(con);
	}

    if (result!=RESULT_MSG_PARTLY_PROCESSED && result != RESULT_SOCKET_CLOSED && result != RESULT_SOCKET_ERR)
	{
		//We can mark the buffer as being empty here safely for following reasons
		// - In case the message has been received completely, it has been already dispatched
		// - In case we got a corrupt message or we ran into some sort of connection error, we are not
		//		interested in the content of the buffer any more
		message_buffer_mark_empty(con->receiver_buffer);
	}
}

static void socket_server_add_connection(int con_socket)
{
    connection_ctx_t *new_con;
    new_con=socket_server_create_connection_ctx(con_socket);
    if (new_con!=NULL)
    {
    	//queue the new connection
    	new_con->next_connection=connection_list_head.next_connection;
        connection_list_head.next_connection=new_con;

        //register it with watch as new event source
        watch_add_event_source(&new_con->watch, EPOLLIN);
    }
    else
        automounter_die_on_resource_issues();
}

static void socket_server_remove_connection(connection_ctx_t *con)
{
    connection_ctx_t *con_itr;

    con_itr=&connection_list_head;
    while (con_itr->next_connection!=NULL)
    {
        if (con_itr->next_connection==con)
        {
            //we found our connected, dequeue it
            con_itr->next_connection=con->next_connection;

            //unregister it from watch
            watch_remove_event_source(&con->watch);
            //close and destroy it
            socket_server_connection_close_and_free_ctx(con);

            break;
        }
        con_itr=con_itr->next_connection;
    }
}

static void socket_server_on_connection_lost(connection_ctx_t *con)
{
	//connection closed, shut it down and remove it
    logger_log_debug("SOCKET_SERVER - Connection to application '%s' has been closed.",
    		socket_server_get_connection_appidentifier(con));
    socket_server_remove_connection(con);
}

static void socket_server_close_all_connections(void)
{
    connection_ctx_t *next_con;

    while(connection_list_head.next_connection!=NULL)
    {
        //safe 2nd connection
        next_con=connection_list_head.next_connection->next_connection;

        //unregister from watch
        watch_remove_event_source(&connection_list_head.next_connection->watch);
        //close and free first connection
        socket_server_connection_close_and_free_ctx(connection_list_head.next_connection);

        //make the 2nd connection the first one
        connection_list_head.next_connection=next_con;
    }
}

static connection_ctx_t *socket_server_create_connection_ctx(int con_socket)
{
    connection_ctx_t *new_con;
    new_con=malloc(sizeof(connection_ctx_t));
    if (new_con!=NULL)
    {
        new_con->watch.pollfd=con_socket;
        new_con->watch.callback_func=socket_server_handle_connection_event;
        new_con->watch.data=new_con;
        new_con->app_identifier=strdup("unknown application");
        new_con->event_mask=0;
        message_buffer_init_queue(&(new_con->sender_queue));
        new_con->receiver_buffer=message_recvr_create_receiver_buffer();
        if (new_con->app_identifier==NULL || new_con->receiver_buffer==NULL)
        	automounter_die_on_resource_issues();
    }
    return new_con;
}

static void socket_server_connection_close_and_free_ctx(connection_ctx_t *con)
{
	if (con==NULL)
		return;
	if (con->watch.pollfd!=-1)
		close(con->watch.pollfd);
	message_sendr_empty_queue(&(con->sender_queue));
	message_recvr_destroy_receiver_buffer(con->receiver_buffer);
	free(con->app_identifier);
	free(con);
}
//--------------------------------------------------------------------------------------------------------
